<?php

namespace App\Http\Controllers\Frontend;

use App\Models\Skill;
use App\Models\JobPost;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Modules\Service\Entities\Category;

class SkillJobController extends Controller
{
    public function skill_jobs($slug)
    {
        $parts = explode('-', $slug, 2);
        $id = $parts[0] ?? null;

        if (is_numeric($id)) {
            $skill = Skill::find($id);

            if (!$skill) {
                return redirect()->route('jobs.all');
            }
        } else {
            $skill = Skill::where('skill', $slug)
                ->first();

            if (!$skill) {
                return redirect()->route('jobs.all');
            }
        }

        // Fetch related jobs
        $query = $skill->jobs()
            ->where('on_off', '1')
            ->where('status', '1')
            ->where('job_approve_request', '1')
            ->withCount('job_proposals')
            ->latest();

        // Apply country restrictions
        $query = $this->applyCountryRestrictions($query);

        $jobs = $query->paginate(10);

        return view('frontend.pages.skill-jobs.jobs', compact(['jobs', 'skill']));
    }

    public function skill_jobs_filter(Request $request)
    {
        if ($request->ajax()) {
            $skill = Skill::select('id', 'skill')->where('id', $request->skill)->first();

            $jobs = $skill->jobs()
                ->where('on_off', '1')
                ->where('status', '1')
                ->withCount('job_proposals')
                ->latest()
                ->where('job_approve_request', '1');

            // Apply country restrictions
            $jobs = $this->applyCountryRestrictions($jobs);

            if (filled($request->job_search_string)) {
                $jobs = $jobs->WhereHas('job_creator')->where('title', 'LIKE', '%' . strip_tags($request->job_search_string) . '%');
            }

            if (isset($request->country) && !empty($request->country)) {
                $jobs = $jobs->WhereHas('job_creator', function ($q) use ($request) {
                    $q->where('country_id', $request->country);
                });
            }
            if (isset($request->type) && !empty($request->type)) {
                $jobs = $jobs->where('type', $request->type);
            }
            if (isset($request->level) && !empty($request->level)) {
                $jobs = $jobs->WhereHas('job_creator', function ($q) use ($request) {
                    $q->where('level', $request->level);
                });
            }
            if (isset($request->min_price) && isset($request->max_price)  && !empty($request->min_price) && !empty($request->max_price)) {
                $jobs = $jobs->whereBetween('budget', [$request->min_price, $request->max_price]);
            }
            if (isset($request->duration) && !empty($request->duration)) {
                $jobs = $jobs->where('duration', $request->duration);
            }

            $jobs = $jobs->paginate(10);
            return $jobs->total() >= 1 ? view('frontend.pages.skill-jobs.search-job-result', compact('jobs'))->render() : response()->json(['status' => __('nothing')]);
        }
    }

    public function pagination(Request $request)
    {
        if ($request->ajax()) {
            $skill = Skill::select('id', 'skill')->where('id', $request->skill)->first();

            $jobs = $skill->jobs()
                ->where('on_off', '1')
                ->where('status', '1')
                ->withCount('job_proposals')
                ->where('job_approve_request', '1');

            // Apply country restrictions
            $jobs = $this->applyCountryRestrictions($jobs);

            if ($request->country == '' && $request->type == '' && $request->level == '' && $request->min_price == '' && $request->max_price == '' && $request->duration == '' && $request->job_search_string == '') {
                $jobs = $jobs;
            } else {
                if (filled($request->job_search_string)) {
                    $jobs = $jobs->WhereHas('job_creator')->where('title', 'LIKE', '%' . strip_tags($request->job_search_string) . '%');
                }

                if (isset($request->country) && !empty($request->country)) {
                    $jobs = $jobs->WhereHas('job_creator', function ($q) use ($request) {
                        $q->where('country_id', $request->country);
                    });
                }
                if (isset($request->type) && !empty($request->type)) {
                    $jobs = $jobs->where('type', $request->type);
                }
                if (isset($request->level) && !empty($request->level)) {
                    $jobs = $jobs->WhereHas('job_creator', function ($q) use ($request) {
                        $q->where('level', $request->level);
                    });
                }
                if (isset($request->min_price) && isset($request->max_price)  && !empty($request->min_price) && !empty($request->max_price)) {
                    $jobs = $jobs->whereBetween('budget', [$request->min_price, $request->max_price]);
                }
                if (isset($request->duration) && !empty($request->duration)) {
                    $jobs = $jobs->where('duration', $request->duration);
                }
            }
            $jobs = $jobs->paginate(10);
            return $jobs->total() >= 1 ? view('frontend.pages.skill-jobs.search-job-result', compact('jobs'))->render() : response()->json(['status' => __('nothing')]);
        }
    }

    //reset jobs filter
    public function reset(Request $request)
    {
        if ($request->ajax()) {
            $skill = Skill::select('id', 'skill')->where('id', $request->skill)->first();

            $query = $skill->jobs()
                ->where('on_off', '1')
                ->where('status', '1')
                ->withCount('job_proposals')
                ->where('job_approve_request', '1')
                ->latest();

            // Apply country restrictions
            $query = $this->applyCountryRestrictions($query);

            $jobs = $query->paginate(10);

            return $jobs->total() >= 1 ? view('frontend.pages.skill-jobs.search-job-result', compact('jobs'))->render() : response()->json(['status' => __('nothing')]);
        } else {
            abort(404);
        }
    }

    /**
     * Apply country restrictions to job query
     */
    private function applyCountryRestrictions($query)
    {
        $restrictionEnabled = get_static_option('job_country_restriction_enabled', 0);
        $viewLevelEnabled = get_static_option('job_country_view_level_enabled', 0);

        if ($restrictionEnabled && $viewLevelEnabled && Auth::check()) {
            $freelancerCountry = Auth::user()->country_id;
            $query = $this->applyCountryRestrictionsToQuery($query, $freelancerCountry);
        }

        return $query;
    }

    /**
     * Helper method to apply country restrictions to query
     */
    private function applyCountryRestrictionsToQuery($query, $freelancerCountry)
    {
        return $query->where(function ($q) use ($freelancerCountry) {
            $q->where(function ($subQ) {
                $subQ->whereNull('country_restriction_type')
                    ->orWhere('country_restriction_type', 'none')
                    ->orWhere('country_restriction_type', '');
            });

            if ($freelancerCountry) {
                $q->orWhere(function ($subQ) use ($freelancerCountry) {
                    $subQ->where('country_restriction_type', 'include')
                        ->whereJsonContains('allowed_countries', (string) $freelancerCountry);
                })
                    ->orWhere(function ($subQ) use ($freelancerCountry) {
                        $subQ->where('country_restriction_type', 'exclude')
                            ->whereJsonDoesntContain('excluded_countries', (string) $freelancerCountry);
                    });
            }
        });
    }
}
