<?php

namespace App\Http\Controllers\Frontend;

use App\Events\ProjectEvent;
use App\Http\Controllers\Controller;
use App\Models\JobPost;
use App\Models\JobProposal;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Intervention\Image\Facades\Image;
use Modules\CurrencySwitcher\App\Models\SelectedCurrencyList;
use Modules\Subscription\Entities\UserSubscription;

class JobDetailsController extends Controller
{
    public function job_details($username = null, $slug = null)
    {
        $job_details = JobPost::with(['job_creator' => function ($q) {
            $q->where('user_active_inactive_status', 1);
        }, 'job_skills', 'job_proposals'])
            ->where('slug', $slug)
            ->first();

        if (!empty($job_details) && $job_details->job_creator) {
            $user = $job_details->job_creator->load('user_country');

            $restrictionEnabled = get_static_option('job_country_restriction_enabled', 0);
            $viewLevelEnabled = get_static_option('job_country_view_level_enabled', 0);
            $canApply = true;
            $canView = true;
            $restrictionMessage = '';
            $requiresProfileUpdate = false;

            if ($restrictionEnabled && Auth::check()) {
                $freelancerCountry = Auth::user()->country_id;

                // Check if this job should be viewable
                $shouldBeViewable = $this->shouldJobBeViewableInListing($job_details, $freelancerCountry);

                if (!$shouldBeViewable && $viewLevelEnabled) {
                    return back()->with(toastr_warning(__('This job is not available in your country.')));
                }

                // Check restrictions for apply functionality
                $restrictionResult = $this->checkCountryRestriction($job_details, $freelancerCountry);

                if (!$restrictionResult['allowed']) {
                    $canApply = false;
                    $restrictionMessage = $restrictionResult['message'];
                    $requiresProfileUpdate = $restrictionResult['requires_profile_update'] ?? false;
                }
            }

            return view('frontend.pages.job-details.job-details', compact(
                'job_details',
                'user',
                'canApply',
                'restrictionMessage',
                'requiresProfileUpdate'
            ));
        }

        return back();
    }

    //job proposal
    public function job_proposal_send(Request $request)
    {
        $request->validate([
            'client_id' => 'required',
            'amount' => 'required|numeric|gt:0',
            'duration' => 'required',
            'revision' => 'required|min:0|max:100',
            'cover_letter' => 'required|min:10|max:1000',
        ]);

        // **ADDED: Prevent restricted words in cover letter**
        if(moduleExists('SecurityManage')) {
            $cover_letter = $request->cover_letter;
            $combinedText = strtolower($cover_letter);

            $restrictedWords = \Modules\SecurityManage\Entities\Word::where('status', 'active')->pluck('word')->toArray();

            $matchedWords = array_filter($restrictedWords, function($word) use ($combinedText) {
                return strpos($combinedText, strtolower($word)) !== false;
            });

            if (count($matchedWords) > 0) {
                return back()->withInput()->with(toastr_error(__('You cannot use restricted words: ') . implode(', ', $matchedWords)));
            }
        }
        // **END OF ADDITION**

        $freelancer_id = Auth::guard('web')->user()->id;
        $check_freelancer_proposal = JobProposal::where('freelancer_id', $freelancer_id)->where('job_id', $request->job_id)->first();
        if ($check_freelancer_proposal) {
            return back()->with(toastr_warning(__('You can not send one more proposal.')));
        }
        if (Auth::guard('web')->user()->is_suspend == 1) {
            return back()->with(toastr_warning(__('You can not send job proposal because your account is suspended. please try to contact admin')));
        }

        // Get job details for country restriction check
        $job_details = JobPost::find($request->job_id);
        if (!$job_details) {
            return back()->with(toastr_error(__('Job not found.')));
        }

        // Check country restrictions
        $restrictionEnabled = get_static_option('job_country_restriction_enabled', 0);
        if ($restrictionEnabled) {
            $freelancerCountry = Auth::user()->country_id;
            $restrictionResult = $this->checkCountryRestriction($job_details, $freelancerCountry);

            if (!$restrictionResult['allowed']) {
                return back()->with(toastr_warning($restrictionResult['message']));
            }
        }

        if (moduleExists('CurrencySwitcher')) {
            $get_user_currency = SelectedCurrencyList::where('currency', get_currency_according_to_user())->first() ?? null;
            $amount = ($request->amount / $get_user_currency->conversion_rate) ?? 0;
            $currency = $get_user_currency->currency ?? 0;
            $conversion_rate = $get_user_currency->conversion_rate ?? 0;
            $symbol = $get_user_currency->symbol ?? null;
        } else {
            $amount = $request->amount;
        }

        if (get_static_option('subscription_enable_disable') != 'disable') {
            $freelancer_subscription = UserSubscription::select(['id', 'user_id', 'limit', 'expire_date', 'created_at'])
                ->where('payment_status', 'complete')
                ->where('status', 1)
                ->where('user_id', $freelancer_id)
                ->where("limit", '>=', get_static_option('limit_settings'))
                ->whereDate('expire_date', '>', Carbon::now())->first();
            $total_limit = UserSubscription::where('user_id', $freelancer_id)->where('payment_status', 'complete')->whereDate('expire_date', '>', Carbon::now())->sum('limit');
            if ($total_limit >= get_static_option('limit_settings') ?? 2 && !empty($freelancer_subscription)) {
                $attachment_name = '';

                $upload_folder = 'jobs/proposal';
                $storage_driver = Storage::getDefaultDriver();
                $extensions = array('png', 'jpg', 'jpeg', 'bmp', 'gif', 'tiff', 'svg');

                if (cloudStorageExist() && in_array(Storage::getDefaultDriver(), ['s3', 'cloudFlareR2', 'wasabi'])) {
                    if ($attachment = $request->file('attachment')) {
                        $request->validate([
                            'attachment' => 'required|mimes:png,jpg,jpeg,bmp,gif,tiff,svg,csv,txt,xlx,xls,pdf,docx|max:2048',
                        ]);
                        $attachment_name = time() . '-' . uniqid() . '.' . $attachment->getClientOriginalExtension();
                        if (in_array($attachment->getClientOriginalExtension(), $extensions)) {
                            add_frontend_cloud_image_if_module_exists($upload_folder, $attachment, $attachment_name, 'public');
                        } else {
                            add_frontend_cloud_image_if_module_exists($upload_folder, $attachment, $attachment_name, 'public');
                        }
                    }
                } else {
                    if ($attachment = $request->file('attachment')) {
                        $request->validate([
                            'attachment' => 'required|mimes:png,jpg,jpeg,bmp,gif,tiff,svg,csv,txt,xlx,xls,pdf,docx|max:2048',
                        ]);
                        $attachment_name = time() . '-' . uniqid() . '.' . $attachment->getClientOriginalExtension();

                        if (in_array($attachment->getClientOriginalExtension(), $extensions)) {
                            $resize_full_image = Image::make($request->attachment)
                                ->resize(1000, 600);
                            $resize_full_image->save('assets/uploads/jobs/proposal' . '/' . $attachment_name);
                        } else {
                            $attachment->move('assets/uploads/jobs/proposal', $attachment_name);
                        }
                    }
                }

                $proposal = JobProposal::create([
                    'job_id' => $request->job_id,
                    'freelancer_id' => auth()->user()->id,
                    'client_id' => $request->client_id,
                    'amount' => $amount,
                    'duration' => $request->duration,
                    'revision' => $request->revision,
                    'cover_letter' => $request->cover_letter,
                    'attachment' => $attachment_name,
                    'currency' => $currency ?? null,
                    'conversion_rate' => $conversion_rate ?? null,
                    'symbol' => $symbol ?? null,
                    'load_from' => in_array($storage_driver, ['CustomUploader']) ? 0 : 1, //added for cloud storage 0=local 1=cloud
                ]);
                client_notification($proposal->id, $request->client_id, 'Proposal', 'You have a new job proposal');
                event(new ProjectEvent(__('You have a new job proposal'), $request->client_id));

                UserSubscription::where('id', $freelancer_subscription->id)->update([
                    'limit' => $freelancer_subscription->limit - (get_static_option('limit_settings') ?? 2)
                ]);

                return back()->with(toastr_success(__('Proposal successfully send')));
            }
            return back()->with(toastr_warning(__('You have not enough connect to apply.')));
        } else {
            $attachment_name = '';
            if ($attachment = $request->file('attachment')) {
                $request->validate([
                    'attachment' => 'required|mimes:png,jpg,jpeg,bmp,gif,tiff,svg,csv,txt,xlx,xls,pdf|max:2048',
                ]);
                $attachment_name = time() . '-' . uniqid() . '.' . $attachment->getClientOriginalExtension();
                $extensions = array('png', 'jpg', 'jpeg', 'bmp', 'gif', 'tiff', 'svg');

                if (in_array($attachment->getClientOriginalExtension(), $extensions)) {
                    $resize_full_image = Image::make($request->attachment)
                        ->resize(1000, 600);
                    $resize_full_image->save('assets/uploads/jobs/proposal' . '/' . $attachment_name);
                } else {
                    $attachment->move('assets/uploads/jobs/proposal', $attachment_name);
                }
            }
            $proposal = JobProposal::create([
                'job_id' => $request->job_id,
                'freelancer_id' => auth()->user()->id,
                'client_id' => $request->client_id,
                'amount' => $amount,
                'duration' => $request->duration,
                'revision' => $request->revision,
                'cover_letter' => $request->cover_letter,
                'currency' => $currency ?? null,
                'conversion_rate' => $conversion_rate ?? null,
                'symbol' => $symbol ?? null,
                'attachment' => $attachment_name,
            ]);
            client_notification($proposal->id, $request->client_id, 'Proposal', __('You have a new job proposal'));
            return back()->with(toastr_success(__('Proposal successfully send')));
        }
    }

    /**
     * Check if job should be viewable in listing
     */
    private function shouldJobBeViewableInListing($job_details, $freelancerCountry)
    {
        // Jobs with no restrictions are always viewable
        if (
            empty($job_details->country_restriction_type) ||
            $job_details->country_restriction_type === 'none' ||
            $job_details->country_restriction_type === ''
        ) {
            return true;
        }

        // If user has no country, they can only see unrestricted jobs
        if (!$freelancerCountry) {
            return false;
        }

        // Apply country-specific logic for users with countries
        if ($job_details->country_restriction_type === 'include') {
            $allowed = $job_details->allowed_countries ?? [];
            return in_array($freelancerCountry, $allowed);
        }

        if ($job_details->country_restriction_type === 'exclude') {
            $excluded = $job_details->excluded_countries ?? [];
            return !in_array($freelancerCountry, $excluded);
        }

        return true;
    }

    /**
     * Helper method to check country restrictions for apply functionality
     */
    private function checkCountryRestriction($job_details, $freelancerCountry)
    {
        if (!$freelancerCountry) {
            return [
                'allowed' => false,
                'message' => __('Please complete your profile with country information to apply for jobs.'),
                'requires_profile_update' => true
            ];
        }

        if (
            empty($job_details->country_restriction_type) ||
            $job_details->country_restriction_type === 'none' ||
            $job_details->country_restriction_type === ''
        ) {
            return ['allowed' => true, 'message' => ''];
        }

        if ($job_details->country_restriction_type === 'include') {
            // Include mode: only allowed countries can access
            $allowed = $job_details->allowed_countries ?? [];
            if (!in_array($freelancerCountry, $allowed)) {
                return [
                    'allowed' => false,
                    'message' => __('This job is only available for specific countries.')
                ];
            }
        }

        if ($job_details->country_restriction_type === 'exclude') {
            // Exclude mode: excluded countries cannot access
            $excluded = $job_details->excluded_countries ?? [];
            if (in_array($freelancerCountry, $excluded)) {
                return [
                    'allowed' => false,
                    'message' => __('This job is not available in your country.')
                ];
            }
        }

        return ['allowed' => true, 'message' => ''];
    }
}
