<?php

namespace App\Http\Controllers\Frontend;

use App\Models\JobPost;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;

class FrontendJobsController extends Controller
{
    public function jobs()
    {
        $query = $this->baseJobQuery();

        $query = $this->applyViewLevelCountryRestriction($query);

        $query = $query->latest();

        if (moduleExists('HourlyJob')) {
            $jobs = $query->paginate(10);
        } else {
            $jobs = $query->where('type', 'fixed')->paginate(10);
        }

        return view('frontend.pages.jobs.jobs', compact('jobs'));
    }

    public function jobs_filter(Request $request)
    {
        if (!$request->ajax()) return abort(404);

        $query = $this->baseJobQuery();

        $query = $this->applyViewLevelCountryRestriction($query);

        if (!moduleExists('HourlyJob')) {
            $query = $query->where('type', 'fixed');
        }

        // Apply other filters
        $query = $this->applyJobFilters($query, $request);

        $jobs = $query->paginate(10);

        return $jobs->total() >= 1
            ? view('frontend.pages.jobs.search-job-result', compact('jobs'))->render()
            : response()->json(['status' => __('nothing')]);
    }

    public function pagination(Request $request)
    {
        if (!$request->ajax()) return abort(404);

        $query = $this->baseJobQuery();

        $query = $this->applyViewLevelCountryRestriction($query);

        if (!moduleExists('HourlyJob')) {
            $query = $query->where('type', 'fixed');
        }

        if (
            $request->job_search_string ||
            $request->category ||
            $request->subcategory ||
            $request->skills ||
            $request->country ||
            $request->state ||
            $request->type ||
            $request->level ||
            $request->min_price ||
            $request->max_price ||
            $request->duration
        ) {
            $query = $this->applyJobFilters($query, $request);
        }

        $jobs = $query->paginate(10);

        return $jobs->total() >= 1
            ? view('frontend.pages.jobs.search-job-result', compact('jobs'))->render()
            : response()->json(['status' => __('nothing')]);
    }

    public function reset(Request $request)
    {
        if (!$request->ajax()) return abort(404);

        $query = $this->baseJobQuery();

        $query = $this->applyViewLevelCountryRestriction($query);

        $query = $query->latest();

        if (!moduleExists('HourlyJob')) {
            $query = $query->where('type', 'fixed');
        }

        $jobs = $query->paginate(10);

        return $jobs->total() >= 1
            ? view('frontend.pages.jobs.search-job-result', compact('jobs'))->render()
            : response()->json(['status' => __('nothing')]);
    }

    // -------------------------
    // Helper Methods
    // -------------------------

    private function baseJobQuery()
    {
        return JobPost::with('job_creator', 'job_skills')
            ->whereHas('job_creator', function ($q) {
                $q->where('user_active_inactive_status', 1);
            })
            ->where('on_off', '1')
            ->withCount('job_proposals')
            ->where('status', '1')
            ->where('job_approve_request', '1');
    }

    private function applyViewLevelCountryRestriction($query)
    {
        $restrictionEnabled = get_static_option('job_country_restriction_enabled', 0);
        $viewLevelEnabled = get_static_option('job_country_view_level_enabled', 0);

        if ($restrictionEnabled && $viewLevelEnabled && Auth::check()) {
            $freelancerCountry = Auth::user()->country_id;
            $query = $this->applyCountryRestrictionsToQuery($query, $freelancerCountry);
        }

        return $query;
    }

    private function applyCountryRestrictionsToQuery($query, $freelancerCountry)
    {
        $query->where(function ($q) use ($freelancerCountry) {
            $q->where(function ($subQ) {
                $subQ->whereNull('country_restriction_type')
                    ->orWhere('country_restriction_type', 'none')
                    ->orWhere('country_restriction_type', '');
            });

            if ($freelancerCountry) {
                $q->orWhere(function ($subQ) use ($freelancerCountry) {
                    $subQ->where('country_restriction_type', 'include')
                        ->whereJsonContains('allowed_countries', (string) $freelancerCountry);
                })
                    ->orWhere(function ($subQ) use ($freelancerCountry) {
                        $subQ->where('country_restriction_type', 'exclude')
                            ->whereJsonDoesntContain('excluded_countries', (string) $freelancerCountry);
                    });
            }
        });

        return $query;
    }

    private function applyJobFilters($query, $request)
    {
        if (filled($request->job_search_string)) {
            $query = $query->where('title', 'LIKE', '%' . strip_tags($request->job_search_string) . '%');
        }

        if (!empty($request->category)) {
            $query = $query->where('category', $request->category);
        }

        if (!empty($request->subcategory)) {
            $query = $query->whereHas('job_sub_categories', function ($q) use ($request) {
                $q->whereIn('sub_categories.id', $request->subcategory);
            });
        }

        if (!empty($request->skills)) {
            $query = $query->whereHas('job_skills', function ($q) use ($request) {
                $q->whereIn('skills.id', $request->skills);
            });
        }

        if (!empty($request->country)) {
            $query = $query->whereHas('job_creator', function ($q) use ($request) {
                $q->where('country_id', $request->country);
            });
        }

        if (!empty($request->state)) {
            $states = is_array($request->state) ? $request->state : [$request->state];
            $query = $query->whereHas('job_creator', function ($q) use ($states) {
                $q->whereIn('state_id', $states);
            });
        }

        if (!empty($request->type)) {
            $query = $query->where('type', $request->type);
        }

        if (!empty($request->level)) {
            $query = $query->whereHas('job_creator', function ($q) use ($request) {
                $q->where('level', $request->level);
            });
        }

        if (!empty($request->min_price) && !empty($request->max_price)) {
            $query = $query->whereBetween('budget', [$request->min_price, $request->max_price]);
        }

        if (!empty($request->duration)) {
            $query = $query->where('duration', $request->duration);
        }
        $query->latest();
        return $query;
    }
}
