<?php

namespace App\Http\Controllers\Frontend;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Pagination\LengthAwarePaginator;

class FreelancerListController extends Controller
{
    private $current_date;

    public function __construct()
    {
        $this->current_date = now()->toDateTimeString();
    }

    // All talents
    public function talents(Request $request)
    {
        $talents = $this->getMixedFreelancers($request);
        return view('frontend.pages.talent.talents', compact('talents'));
    }

    // Talents pagination
    public function pagination(Request $request)
    {
        if ($request->ajax()) {
            $talents = $this->getMixedFreelancers($request);
            $this->trackProFreelancerImpressions($talents);

            return $talents->total() >= 1
                ? view('frontend.pages.talent.search-talent-result', compact('talents'))->render()
                : response()->json(['status' => __('nothing')]);
        }
    }

    // Talents filter
    public function talents_filter(Request $request)
    {
        if ($request->ajax()) {
            $talents = $this->getMixedFreelancers($request);
            $this->trackProFreelancerImpressions($talents);

            return $talents->total() >= 1
                ? view('frontend.pages.talent.search-talent-result', compact('talents'))->render()
                : response()->json(['status' => __('nothing')]);
        }
    }

    // Reset filter
    public function reset(Request $request)
    {
        $talents = $this->getMixedFreelancers($request);
        return $talents->total() >= 1
            ? view('frontend.pages.talent.search-talent-result', compact('talents'))->render()
            : response()->json(['status' => __('nothing')]);
    }

    // Get mixed pro/non-pro freelancers
    private function getMixedFreelancers($request)
    {
        $perPage = get_static_option("projects_per_page") ?? 12;
        $proCount = get_static_option("pro_projects_count") ?? 6;
        $nonProCount = get_static_option("non_pro_projects_count") ?? 6;
        $proFirst = get_static_option("pro_projects_default_first") == 0;
        $page = $request->get('page', 1);
        $hasPromo = moduleExists('PromoteFreelancer');

        // Ensure pro + non-pro counts equal per page
        $totalConfigured = $proCount + $nonProCount;
        if ($totalConfigured !== $perPage) {
            // Adjust proportionally
            $proCount = (int) round(($proCount / $totalConfigured) * $perPage);
            $nonProCount = $perPage - $proCount;
        }

        if (!$hasPromo) {
            return $this->getFreelancersWithoutPromotion($request, $perPage);
        }

        $baseQuery = $this->filter_query($request);

        $currentProIds = User::where('is_pro', 'yes')
            ->where('pro_expire_date', '>=', $this->current_date)
            ->pluck('id')
            ->toArray();

        if ($proFirst && !empty($currentProIds)) {
            return $this->getProFreelancersFirstOptimized($baseQuery, $currentProIds, $page, $perPage, $request);
        }

        return $this->getMixedFreelancersOptimized($baseQuery, $currentProIds, $page, $perPage, $proCount, $nonProCount, $request);
    }

    // Freelancers without promotion module
    private function getFreelancersWithoutPromotion($request, $perPage)
    {
        $query = $this->filter_query($request);
        return $query->orderBy('freelancer_orders_count', 'desc')->paginate($perPage);
    }

    // Pro freelancers first
    private function getProFreelancersFirstOptimized($baseQuery, $proIds, $page, $perPage, $request)
    {
        // Get total counts first
        $totalPro = (clone $baseQuery)->whereIn('id', $proIds)->count();
        $totalNonPro = (clone $baseQuery)->whereNotIn('id', $proIds)->count();
        $totalItems = $totalPro + $totalNonPro;

        if ($totalItems == 0) {
            return new LengthAwarePaginator(
                collect([]),
                0,
                $perPage,
                $page,
                ['path' => request()->url(), 'query' => request()->query()]
            );
        }

        $offset = ($page - 1) * $perPage;
        $result = collect();

        // If we're still in pro range
        if ($offset < $totalPro) {
            $proTake = min($perPage, $totalPro - $offset);
            $proFreelancers = (clone $baseQuery)
                ->whereIn('id', $proIds)
                ->orderByRaw('RAND(' . $this->getConsistentSeed($request) . ')')
                ->offset($offset)
                ->limit($proTake)
                ->get();

            $result = $result->concat($proFreelancers);
        }

        // If we need non-pro to fill the page
        if ($result->count() < $perPage && $offset + $result->count() >= $totalPro) {
            $nonProOffset = max(0, $offset - $totalPro);
            $nonProTake = $perPage - $result->count();

            $nonProFreelancers = (clone $baseQuery)
                ->whereNotIn('id', $proIds)
                ->orderBy('freelancer_orders_count', 'desc')
                ->offset($nonProOffset)
                ->limit($nonProTake)
                ->get();

            $result = $result->concat($nonProFreelancers);
        }

        return new LengthAwarePaginator(
            $result,
            $totalItems,
            $perPage,
            $page,
            ['path' => request()->url(), 'query' => request()->query()]
        );
    }

    // Mixed freelancers with ratio
    private function getMixedFreelancersOptimized($baseQuery, $proIds, $page, $perPage, $proCount, $nonProCount, $request)
    {
        $adjustedCounts = $this->adjustCountsForAvailability($baseQuery, $proIds, $perPage, $proCount, $nonProCount);

        $actualProCount = $adjustedCounts['proCount'];
        $actualNonProCount = $adjustedCounts['nonProCount'];
        $totalPro = $adjustedCounts['totalAvailablePro'];
        $totalNonPro = $adjustedCounts['totalAvailableNonPro'];

        $totalItems = $totalPro + $totalNonPro;

        if ($totalItems == 0) {
            return new LengthAwarePaginator(
                collect([]),
                0,
                $perPage,
                $page,
                ['path' => request()->url(), 'query' => request()->query()]
            );
        }

        // Calculate what we need for current page using adjusted counts
        $pageData = $this->calculatePageAllocation($page, $perPage, $actualProCount, $actualNonProCount, $totalPro, $totalNonPro);

        $result = collect();

        // Get pro freelancers for current page with consistent ordering
        if ($pageData['proTake'] > 0) {
            $proFreelancers = (clone $baseQuery)
                ->whereIn('id', $proIds)
                ->orderByRaw('RAND(' . $this->getConsistentSeed($request) . ')')
                ->offset($pageData['proOffset'])
                ->limit($pageData['proTake'])
                ->get();
            $result = $result->concat($proFreelancers->keyBy('id'));
        }

        // Get non-pro freelancers for current page
        if ($pageData['nonProTake'] > 0) {
            $nonProFreelancers = (clone $baseQuery)
                ->whereNotIn('id', $proIds)
                ->orderBy('freelancer_orders_count', 'desc')
                ->offset($pageData['nonProOffset'])
                ->limit($pageData['nonProTake'])
                ->get();
            $result = $result->concat($nonProFreelancers->keyBy('id'));
        }

        // Interleave the results using adjusted counts
        $mixed = $this->interleaveResults($result, $proIds, $actualProCount, $actualNonProCount);

        return new LengthAwarePaginator(
            $mixed,
            $totalItems,
            $perPage,
            $page,
            ['path' => request()->url(), 'query' => request()->query()]
        );
    }

    // Adjust counts for availability
    private function adjustCountsForAvailability($baseQuery, $proIds, $perPage, $proCount, $nonProCount)
    {
        // Get actual available counts
        $totalAvailablePro = (clone $baseQuery)->whereIn('id', $proIds)->count();
        $totalAvailableNonPro = (clone $baseQuery)->whereNotIn('id', $proIds)->count();

        // Adjust counts based on availability
        $actualProCount = min($proCount, $totalAvailablePro);
        $actualNonProCount = min($nonProCount, $totalAvailableNonPro);

        // Fill remaining slots to maintain perPage count
        $totalConfigured = $actualProCount + $actualNonProCount;
        $remainingSlots = $perPage - $totalConfigured;

        if ($remainingSlots > 0) {
            $extraNonPro = min($remainingSlots, $totalAvailableNonPro - $actualNonProCount);
            $actualNonProCount += $extraNonPro;
            $remainingSlots -= $extraNonPro;

            // add more pro if still needed
            if ($remainingSlots > 0) {
                $extraPro = min($remainingSlots, $totalAvailablePro - $actualProCount);
                $actualProCount += $extraPro;
            }
        }

        return [
            'proCount' => $actualProCount,
            'nonProCount' => $actualNonProCount,
            'totalAvailablePro' => $totalAvailablePro,
            'totalAvailableNonPro' => $totalAvailableNonPro
        ];
    }

    // Calculate page allocation for mixed results
    private function calculatePageAllocation($page, $perPage, $proCount, $nonProCount, $totalPro, $totalNonPro)
    {
        $proUsed = 0;
        $nonProUsed = 0;

        // Calculate allocation for previous pages
        for ($p = 1; $p < $page; $p++) {
            $pagePro = min($proCount, $totalPro - $proUsed);
            $pageNonPro = min($nonProCount, $totalNonPro - $nonProUsed);
            $pageTotal = $pagePro + $pageNonPro;

            // Fill remaining slots
            if ($pageTotal < $perPage) {
                $remaining = $perPage - $pageTotal;

                // Try to add more non-pro first
                $extraNonPro = min($remaining, $totalNonPro - $nonProUsed - $pageNonPro);
                $pageNonPro += $extraNonPro;
                $remaining -= $extraNonPro;

                // Then add more pro if needed
                if ($remaining > 0) {
                    $extraPro = min($remaining, $totalPro - $proUsed - $pagePro);
                    $pagePro += $extraPro;
                }
            }

            $proUsed += $pagePro;
            $nonProUsed += $pageNonPro;
        }

        // Calculate for current page
        $currentPagePro = min($proCount, $totalPro - $proUsed);
        $currentPageNonPro = min($nonProCount, $totalNonPro - $nonProUsed);
        $currentPageTotal = $currentPagePro + $currentPageNonPro;

        // Fill remaining slots for current page
        if ($currentPageTotal < $perPage) {
            $remaining = $perPage - $currentPageTotal;

            // Try to add more non-pro first
            $extraNonPro = min($remaining, $totalNonPro - $nonProUsed - $currentPageNonPro);
            $currentPageNonPro += $extraNonPro;
            $remaining -= $extraNonPro;

            // Then add more pro if needed
            if ($remaining > 0) {
                $extraPro = min($remaining, $totalPro - $proUsed - $currentPagePro);
                $currentPagePro += $extraPro;
            }
        }

        return [
            'proOffset' => $proUsed,
            'proTake' => $currentPagePro,
            'nonProOffset' => $nonProUsed,
            'nonProTake' => $currentPageNonPro
        ];
    }

    // Interleave results with scattered distribution (most effective)
    private function interleaveResults($allResults, $proIds, $proCount, $nonProCount)
    {
        $proResults = $allResults->whereIn('id', $proIds)->values();
        $nonProResults = $allResults->whereNotIn('id', $proIds)->values();

        $totalItems = $proResults->count() + $nonProResults->count();
        if ($totalItems === 0) return collect();

        // Handle edge cases
        if ($proResults->count() === 0) return $nonProResults;
        if ($nonProResults->count() === 0) return $proResults;

        // Better visibility for paid promotions
        return $this->simplePromotionMix($proResults, $nonProResults);
    }

    private function simplePromotionMix($proResults, $nonProResults)
    {
        $proCount = $proResults->count();

        if ($proCount <= 3) {
            // For few promoted: Place them strategically with good spacing
            return $this->fewPromotedStrategy($proResults, $nonProResults);
        } else {
            // For many promoted: Distribute more evenly but still favor early positions
            return $this->manyPromotedStrategy($proResults, $nonProResults);
        }
    }

    // Better strategic positioning for promoted freelancers
    private function fewPromotedStrategy($proResults, $nonProResults)
    {
        $mixed = collect();
        $proIndex = 0;
        $nonProIndex = 0;
        $totalItems = $proResults->count() + $nonProResults->count();
        $proCount = $proResults->count();

        $promotedPositions = [];

        if ($proCount == 1) {
            $promotedPositions = [0]; // Position 1
        } elseif ($proCount == 2) {
            $promotedPositions = [0, 5]; // Positions 1, 6
        } elseif ($proCount == 3) {
            $promotedPositions = [0, 2, 6]; // Positions 1, 3, 7
        } else {
            // For more than 3, use calculated spacing
            $spacing = max(3, intval($totalItems / $proCount));
            for ($i = 0; $i < $proCount; $i++) {
                $pos = $i * $spacing;
                if ($pos < $totalItems) {
                    $promotedPositions[] = $pos;
                }
            }
        }

        for ($i = 0; $i < $totalItems; $i++) {
            // Place promoted at strategic positions
            if (in_array($i, $promotedPositions) && $proIndex < $proResults->count()) {
                $mixed->push($proResults[$proIndex]);
                $proIndex++;
            } else if ($nonProIndex < $nonProResults->count()) {
                $mixed->push($nonProResults[$nonProIndex]);
                $nonProIndex++;
            } else if ($proIndex < $proResults->count()) {
                // Fill remaining with promoted if no more non-promoted
                $mixed->push($proResults[$proIndex]);
                $proIndex++;
            }
        }

        return $mixed->filter()->values();
    }

    private function manyPromotedStrategy($proResults, $nonProResults)
    {
        $mixed = collect();
        $proIndex = 0;
        $nonProIndex = 0;
        $totalItems = $proResults->count() + $nonProResults->count();
        $proCount = $proResults->count();

        // Calculate positions that give good spacing: 1, 3, 6, 8, 10, 12
        $promotedPositions = [];

        if ($proCount >= 4) {
            // For 4-6 promoted: Use alternating pattern with gaps
            // Pattern: 1, 3, 6, 8, 10, 12 (positions 0, 2, 5, 7, 9, 11)
            $strategicPositions = [0, 2, 5, 7, 9, 11];
            $promotedPositions = array_slice($strategicPositions, 0, min($proCount, count($strategicPositions)));
        } else {
            // Fallback for edge cases
            $spacing = max(2, intval($totalItems / $proCount));
            for ($i = 0; $i < $proCount; $i++) {
                $pos = $i * $spacing;
                if ($pos < $totalItems) {
                    $promotedPositions[] = $pos;
                }
            }
        }

        for ($i = 0; $i < $totalItems; $i++) {
            // Place promoted at strategic positions
            if (in_array($i, $promotedPositions) && $proIndex < $proResults->count()) {
                $mixed->push($proResults[$proIndex]);
                $proIndex++;
            } else if ($nonProIndex < $nonProResults->count()) {
                $mixed->push($nonProResults[$nonProIndex]);
                $nonProIndex++;
            } else if ($proIndex < $proResults->count()) {
                // Fill remaining with promoted if no more non-promoted
                $mixed->push($proResults[$proIndex]);
                $proIndex++;
            }
        }

        return $mixed->filter()->values();
    }

    // Track pro freelancer impressions
    private function trackProFreelancerImpressions($freelancers)
    {
        if (!moduleExists('PromoteFreelancer')) return;

        $authId = auth('web')->id();

        $proFreelancerIds = collect();

        foreach ($freelancers as $freelancer) {
            if (!empty($freelancer->is_pro_freelancer) && $freelancer->id !== $authId) {
                $proFreelancerIds->push($freelancer->id);
            }
        }

        if ($proFreelancerIds->isNotEmpty()) {
            // Bulk update impressions for better performance
            \Modules\PromoteFreelancer\Entities\PromotionProjectList::whereIn('identity', $proFreelancerIds)
                ->where('type', 'profile')
                ->where('expire_date', '>=', $this->current_date)
                ->increment('impression');
        }
    }


    /**
     * Generate consistent seed for randomization across pagination
     * Create seed based on search filters + session to maintain consistency within session
     * but allow variation between different sessions/users 
     * 
     */
    private function getConsistentSeed($request)
    {
        // Handle skill as array or string
        $skillValue = '';
        if (!empty($request->skill)) {
            $skillValue = is_array($request->skill)
                ? implode(',', $request->skill)
                : $request->skill;
        }

        // Handle subcategory as array or string
        $subcategoryValue = '';
        if (!empty($request->subcategory)) {
            $subcategoryValue = is_array($request->subcategory)
                ? implode(',', $request->subcategory)
                : $request->subcategory;
        }

        $seedData = [
            $request->job_search_string ?? '',
            $request->gender ?? '',
            $request->country ?? '',
            $request->level ?? '',
            $request->category ?? '',
            $request->subcategory ?? '',
            $skillValue,
            $request->min_count ?? '',
            $request->max_count ?? '',
            $request->talent_badge ?? '',
            session()->getId(),
            date('H')
        ];

        return crc32(implode('|', $seedData));
    }

    // filter query
    private function filter_query($request)
    {
        $talents = User::query()
            ->select('id', 'username', 'first_name', 'last_name', 'image', 'country_id', 'state_id', 'is_pro', 'pro_expire_date', 'user_verified_status', 'load_from', 'created_at')
            ->with('user_introduction')
            ->where('user_type', '2')
            ->where('is_email_verified', 1)
            ->where('is_suspend', 0)
            ->where('user_active_inactive_status', 1)
            ->where('check_work_availability', 1)
            ->withCount(['freelancer_orders' => function ($query) {
                $query->where('status', 3);
            }])
            ->withSum(['freelancer_orders' => function ($query) {
                $query->where('status', 3);
            }], 'payable_amount')
            ->withAvg(['freelancer_ratings' => function ($query) {
                $query->where('sender_type', 1);
            }], 'rating')
            ->orderBy('freelancer_orders_count', 'DESC');

        if (!empty($request->country)) {
            $talents = $talents->where('country_id', $request->country);
        }

        if (!empty($request->level)) {
            $talents = $talents->where('experience_level', $request->level);
        }

        if (!empty($request->category)) {
            $talents = $talents->whereHas('freelancer_category', function ($q) use ($request) {
                $q->where('category_id', $request->category);
            });
        }

        if (!empty($request->subcategory)) {
            $talents = $talents->whereHas('freelancer_subcategory', function ($q) use ($request) {
                $q->whereIn('sub_category_id', $request->subcategory);
            });
        }

        // Even better solution - replace the talent_badge filtering section with this:

        if (!empty($request->talent_badge)) {
            $rule = \Modules\FreelancerLevel\Entities\FreelancerLevelRules::where('period', $request->talent_badge)->first();

            if (!$rule) {
                return $talents;
            }

            $minDays = null;
            $maxDays = null;

            if ($request->talent_badge >= 1 && $request->talent_badge < 3) {
                $minDays = 30;
                $maxDays = 90;
            } elseif ($request->talent_badge >= 3 && $request->talent_badge < 6) {
                $minDays = 90;
                $maxDays = 180;
            } elseif ($request->talent_badge >= 6 && $request->talent_badge < 9) {
                $minDays = 180;
                $maxDays = 270;
            } elseif ($request->talent_badge >= 9 && $request->talent_badge < 12) {
                $minDays = 270;
                $maxDays = 360;
            } elseif ($request->talent_badge >= 12) {
                $minDays = 360;
                $maxDays = null;
            }

            if ($minDays !== null) {
                $maxDate = now()->subDays($minDays);
                $minDate = $maxDays ? now()->subDays($maxDays) : null;

                if ($minDate) {
                    $talents = $talents
                        ->whereDate('created_at', '>=', $minDate)
                        ->whereDate('created_at', '<=', $maxDate);
                } else {
                    $talents = $talents->whereDate('created_at', '<=', $maxDate);
                }

                $talents = $talents->whereExists(function ($query) use ($rule) {
                    $query->select(DB::raw(1))
                        ->from('orders')
                        ->whereColumn('orders.freelancer_id', 'users.id')
                        ->where('orders.status', 3)
                        ->groupBy('orders.freelancer_id')
                        ->havingRaw('COUNT(*) >= ?', [$rule->complete_order])
                        ->havingRaw('SUM(orders.payable_amount) >= ?', [$rule->earning]);
                });

                $talents = $talents->whereExists(function ($query) use ($rule) {
                    $query->select(DB::raw(1))
                        ->from('ratings')
                        ->join('orders', 'orders.id', '=', 'ratings.order_id')
                        ->whereColumn('orders.freelancer_id', 'users.id')
                        ->where('orders.status', 3)
                        ->where('ratings.sender_type', 1)
                        ->groupBy('orders.freelancer_id')
                        ->havingRaw('AVG(ratings.rating) >= ?', [$rule->avg_rating]);
                });
            }
        }

        if (!empty($request->skill)) {
            $skills = is_array($request->skill) ? $request->skill : [$request->skill];
            $skills = array_map('trim', $skills);

            $talents = $talents->whereHas('freelancer_skill', function ($q) use ($skills) {
                foreach ($skills as $index => $skill) {
                    if ($index === 0) {
                        $q->where('skill', 'LIKE', "%{$skill}%");
                    } else {
                        $q->orWhere('skill', 'LIKE', "%{$skill}%");
                    }
                }
            });
        }

        return $talents->orderBy('check_online_status', 'DESC');
    }
}
